// ---------------------------------------------------------------------------
//
//  (c) Copyright 2010, Allan Petersen
//  ALL RIGHTS RESERVED
//  Permission to use, copy, modify, and distribute this software for
//  any purpose and without fee is hereby granted, provided that the above
//  copyright notice appear in all copies and that both the copyright notice
//  and this permission notice appear in supporting documentation, and that
//  the name of Allan Petersen not be used in advertising
//  or publicity pertaining to distribution of the software without specific,
//  written prior permission.
//
//  THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
//  AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
//  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
//  FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL ALLAN
//  PETERSEN BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
//  SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
//  KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
//  LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
//  THIRD PARTIES, WHETHER OR NOT ALLAN PETERSEN HAS BEEN
//  ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
//  ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
//  POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  www.allanpetersen.com
//
//
// ---------------------------------------------------------------------------
// This module requires Windows Vista or newer to operate..
// ---------------------------------------------------------------------------

#include <vcl.h>
#pragma hdrstop

#include "c_uac.h"

// ---------------------------------------------------------------------------

#pragma package(smart_init)

// ---------------------------------------------------------------------------
__fastcall TUAC::TUAC()
{
	if (!CheckWin32Version(6, 0)) {
		throw Exception (AnsiString(__FUNC__) + " This is not Windows Vista or never");
		}

	hToken = NULL;
	pTokenLabel = NULL;
	// Required to query an access token.
	if (!OpenProcessToken(GetCurrentProcess(), TOKEN_QUERY | TOKEN_DUPLICATE,
			&hToken))
	{
		throw GetLastError();
	}
};

// ---------------------------------------------------------------------------
__fastcall TUAC::~TUAC()
{
	if (hToken)
	{
		CloseHandle(hToken);
		hToken = NULL;
	}
	if (pTokenLabel)
	{
		LocalFree(pTokenLabel);
		pTokenLabel = NULL;
	}
};

//===========================================================================
///
///	UAC Administrator status
///	\return	true Admin
///	\return	false Not Admin
///
//---------------------------------------------------------------------------
//	2010-04-07
//===========================================================================
bool __fastcall TUAC::IsRunAsAdmin()
{
	BOOL FIsRunAsAdmin = false;
	PSID pAdministratorsGroup = NULL;
	SID_IDENTIFIER_AUTHORITY NtAuthority = SECURITY_NT_AUTHORITY;

	if (!AllocateAndInitializeSid(&NtAuthority, 2, SECURITY_BUILTIN_DOMAIN_RID,
			DOMAIN_ALIAS_RID_ADMINS, 0, 0, 0, 0, 0, 0, &pAdministratorsGroup))
	{
		throw Exception (AnsiString(__FUNC__) + " Initialize SID failed");
	}

	if (!CheckTokenMembership(NULL, pAdministratorsGroup, &FIsRunAsAdmin))
	{
		FIsRunAsAdmin = false;
	}

	if (pAdministratorsGroup)
	{
		FreeSid(pAdministratorsGroup);
		pAdministratorsGroup = NULL;
	}

	return FIsRunAsAdmin;
}

//===========================================================================
///
///	Validated member of administrator group
///	\return	true In Admin Group
///	\return	false Not in Admin Group
///
//---------------------------------------------------------------------------
//	2010-04-07
//===========================================================================
bool __fastcall TUAC::IsUserInAdminGroup()
{
	BOOL 	FInAdminGroup = FALSE;
	HANDLE 	hTokenToCheck = NULL;
	DWORD 	Size = 0;
	BYTE 	AdminSID[SECURITY_MAX_SID_SIZE];
	TOKEN_ELEVATION_TYPE ElevevationType;

	// Determine token type (default, Full, Limited)
	if (!GetTokenInformation(hToken, TokenElevationType, &ElevevationType,
			sizeof(ElevevationType), &Size))
	{
		throw Exception (AnsiString(__FUNC__) + " Get Token Information Failed");
	}

	// If limited, get the linked elevated token
	if (TokenElevationTypeLimited == ElevevationType)
	{
		if (!GetTokenInformation(hToken, TokenLinkedToken, &hTokenToCheck,
				sizeof(hTokenToCheck), &Size))
		{
			throw Exception (AnsiString(__FUNC__) + " Get Limited Token Information Failed");
		}
	}

	if (!hTokenToCheck)
	{
		if (!DuplicateToken(hToken, SecurityIdentification, &hTokenToCheck))
		{
			throw Exception (AnsiString(__FUNC__) + " Duplicate Token Information Failed");
		}
	}

	Size = sizeof(AdminSID);
	if (!CreateWellKnownSid(WinBuiltinAdministratorsSid, NULL, &AdminSID,
			&Size))
	{
		if (hTokenToCheck)
		{
			CloseHandle(hTokenToCheck);
			hTokenToCheck = NULL;
		}
		return false;
	}

	if (!CheckTokenMembership(hTokenToCheck, &AdminSID, &FInAdminGroup))
	{
		FInAdminGroup = false;
	}

	if (hTokenToCheck)
	{
		CloseHandle(hTokenToCheck);
		hTokenToCheck = NULL;
	}
	return FInAdminGroup;
}

//===========================================================================
// There are four primary integrity levels in Windows Vista with four corresponding values.
// A lower value indicates a lower integrity level or a lower level of trustworthiness.
// These values are defined in the header file, winnt.h.
//
// Value  Description  Symbol
// 0x0000
// Untrusted level
// SECURITY_MANDATORY_UNTRUSTED_RID
//
// 0x1000
// Low integrity level
// SECURITY_MANDATORY_LOW_RID
//
// 0x2000
// Medium integrity level
// SECURITY_MANDATORY_MEDIUM_RID
//
// 0x3000
// High integrity level
// SECURITY_MANDATORY_HIGH_RID
//
// 0x4000
// System integrity level
// SECURITY_MANDATORY_SYSTEM_RID
///
///	\return	Integrity Level
///
//---------------------------------------------------------------------------
//	2010-04-07
//===========================================================================
int __fastcall TUAC::GetIntegrityLevel()
{
	DWORD TokenLevel = 0;

	if (!GetTokenInformation(hToken, TokenIntegrityLevel, NULL, 0, &TokenLevel))
	{
		if (ERROR_INSUFFICIENT_BUFFER != GetLastError())
		{
			throw Exception (AnsiString(__FUNC__) + " Get Token Information Failed");
//			return -1;
		}
	}
	pTokenLabel = (TOKEN_MANDATORY_LABEL*)LocalAlloc(LPTR, TokenLevel);
	if (pTokenLabel == NULL)
	{
		throw Exception (AnsiString(__FUNC__) + " Allocation of Token Information Failed");
	}
	if (!GetTokenInformation(hToken, TokenIntegrityLevel, pTokenLabel, TokenLevel,
			&TokenLevel))
	{
			throw Exception (AnsiString(__FUNC__) + " Get Token Information Failed");
	}
	return *GetSidSubAuthority(pTokenLabel->Label.Sid, 0);
}

//===========================================================================
///
///	Retrieves UAC elaevation level
///	\return	true Elevated
///	\return	false Not Elevated
///
//---------------------------------------------------------------------------
//	2010-04-07
//===========================================================================
bool __fastcall TUAC::IsElevated()
{
	TOKEN_ELEVATION Elevation;
	DWORD Size;

	// Retrieve token elevation information.
	if (!GetTokenInformation(hToken, TokenElevation, &Elevation,
			sizeof(TOKEN_ELEVATION), &Size))
	{
		throw Exception (AnsiString(__FUNC__) + " Get Token Information Failed");
	}
	return Elevation.TokenIsElevated;
}
